<?php

namespace WPFormsCalculations\Admin;

use WPFormsCalculations\Transpiler\Transpiler;

/**
 * Calculation AJAX in Form Builder.
 *
 * @since 1.0.0
 */
class Ajax {

	/**
	 * Init.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		$this->hooks();
	}

	/**
	 * Hooks.
	 *
	 * @since 1.0.0
	 */
	private function hooks() {

		add_filter( 'wpforms_save_form_args', [ $this, 'save_form_args' ], 10, 3 );
		add_action( 'wp_ajax_wpforms_calculations_validate_formula',  [ $this, 'validate_formula' ] );
	}

	/**
	 * Save form args action callback.
	 * Process all the fields with enabled calculations.
	 *
	 * @since 1.0.0
	 *
	 * @param array $form Form update post arguments.
	 * @param array $data Initial form data.
	 * @param array $args Update form args.
	 *
	 * @return array
	 * @noinspection PhpUnusedParameterInspection
	 */
	public function save_form_args( $form, $data, $args ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed

		$form_data = json_decode( stripslashes( $form['post_content'] ), true );

		if ( ! wpforms_calculations()->helpers->has_calculation_enabled_field( $form_data ) ) {
			return $form;
		}

		$transpiler = new Transpiler();

		foreach ( $form_data['fields'] as $field_id => $field ) {

			if ( empty( $field['calculation_is_enabled'] ) ) {
				continue;
			}

			$form_data['fields'][ $field_id ] = $transpiler->process_field( $field, $form_data );
		}

		$form['post_content'] = wpforms_encode( $form_data );

		return $form;
	}

	/**
	 * Validate formula AJAX callback.
	 *
	 * @since 1.0.0
	 */
	public function validate_formula() { // phpcs:ignore Generic.Metrics.CyclomaticComplexity.TooHigh

		// Run a security check.
		if ( ! check_ajax_referer( 'wpforms-builder', 'nonce', false ) ) {
			wp_send_json_error( esc_html__( 'Your session expired. Please reload the builder.', 'wpforms-calculations' ) );
		}

		// Check for permissions.
		if ( ! wpforms_current_user_can( 'edit_forms' ) ) {
			wp_send_json_error( esc_html__( 'You are not allowed to perform this action.', 'wpforms-calculations' ) );
		}

		$form_obj = wpforms()->obj( 'form' );

		// Check for form ID and field ID.
		if ( empty( $_POST['form_id'] ) || ! isset( $_POST['field_id'] ) || ! $form_obj ) {
			wp_send_json_error( esc_html__( 'Something went wrong while performing this action.', 'wpforms-calculations' ) );
		}

		$form_id   = (int) $_POST['form_id'];
		$field_id  = (int) $_POST['field_id'];
		$form_data = $form_obj->get( $form_id, [ 'content_only' => true ] );

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$code       = isset( $_POST['code'] ) ? trim( wp_unslash( $_POST['code'] ) ) : '';
		$transpiler = new Transpiler();

		// If the form is still not saved, we have to mimic the field in the $form_data array.
		// It is needed just to be able to run the code validation.
		if ( ! isset( $form_data['fields'][ $field_id ] ) ) {
			$form_data['fields'][ $field_id ] = [
				'id' => $field_id,
			];
		}

		$errors = $transpiler->parse_and_validate_formula_code( $code, $field_id, $form_data, false );

		if ( ! empty( $errors ) ) {
			wp_send_json_error( $errors );
			die();
		}

		wp_send_json_success();
	}
}
