<?php

namespace WPFormsCalculations\Admin;

use WPForms_Field;
use WPFormsCalculations\Helpers;
use WPFormsCalculations\Transpiler\Functions;

/**
 * Calculation feature.
 *
 * @since 1.0.0
 */
class Builder {

	/**
	 * Script handle.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	const SCRIPT_HANDLE = 'wpforms-calculations-builder';

	/**
	 * Modules handle.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	const MODULES_HANDLE = 'wpforms-calculations-builder-modules';

	/**
	 * Support calculations in these field types.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	const ALLOWED_FIELD_TYPES = [ 'text', 'textarea', 'number', 'hidden', 'payment-single' ];

	/**
	 * Helpers class instance.
	 *
	 * @since 1.0.0
	 *
	 * @var Helpers
	 */
	private $helpers;

	/**
	 * Functions class instance.
	 *
	 * @since 1.0.0
	 *
	 * @var Functions
	 */
	private $functions;

	/**
	 * Init.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		$this->helpers   = wpforms_calculations()->helpers;
		$this->functions = new Functions();

		$this->hooks();
	}

	/**
	 * Hooks.
	 *
	 * @since 1.0.0
	 */
	private function hooks() {

		add_action( 'wpforms_builder_enqueues', [ $this, 'builder_css' ] );
		add_action( 'wpforms_builder_enqueues', [ $this, 'builder_js' ] );
		add_action( 'wpforms_field_options_bottom_advanced-options', [ $this, 'advanced_options' ], 50, 2 );
	}

	/**
	 * Enqueue builder CSS.
	 *
	 * @since 1.0.0
	 */
	public function builder_css() {

		$min = wpforms_get_min_suffix();

		wp_enqueue_style(
			self::SCRIPT_HANDLE,
			WPFORMS_CALCULATIONS_URL . "assets/css/builder{$min}.css",
			[],
			WPFORMS_CALCULATIONS_VERSION
		);
	}

	/**
	 * Enqueue builder js.
	 *
	 * @since 1.0.0
	 */
	public function builder_js() {

		$min = wpforms_get_min_suffix();

		wp_enqueue_code_editor( [ 'type' => 'text/php' ] );

		wp_enqueue_script(
			self::MODULES_HANDLE,
			WPFORMS_CALCULATIONS_URL . "assets/js/builder/modules.es5{$min}.js",
			[ 'jquery', 'wpforms-builder' ],
			WPFORMS_CALCULATIONS_VERSION,
			true
		);

		wp_enqueue_script(
			self::SCRIPT_HANDLE,
			WPFORMS_CALCULATIONS_URL . "assets/js/builder/builder{$min}.js",
			[ 'jquery', 'wpforms-builder', self::MODULES_HANDLE ],
			WPFORMS_CALCULATIONS_VERSION,
			true
		);

		// Localize data.
		wp_localize_script(
			self::SCRIPT_HANDLE,
			'wpforms_calculations',
			[
				'allowedFields'               => $this->helpers->get_fields_allowed_in_calculation(),
				'calculationIsPossibleFileds' => $this->helpers->get_fields_calculation_is_possible(),
				'functionsList'               => $this->functions->get_names(),
				'strings'                     => [
					'expand'                           => esc_html__( 'Expand Editor', 'wpforms-calculations' ),
					'collapse'                         => esc_html__( 'Collapse Editor', 'wpforms-calculations' ),
					'insertFieldDropdownTitle'         => esc_html__( 'Form Fields', 'wpforms-calculations' ),
					'validateButtonSuccess'            => esc_html__( 'The code is correct. Click to re-validate.', 'wpforms-calculations' ),
					'validateButtonErrors'             => esc_html__( 'The code has errors. Click to re-validate.', 'wpforms-calculations' ),
					'validateButtonAjaxError'          => esc_html__( 'The validation attempt failed, there was a problem communicating with the server', 'wpforms-calculations' ),
					'validationModalErrorTitle'        => esc_html__( 'Uh oh!', 'wpforms-calculations' ),
					'validationModalErrorMsg'          => esc_html__( 'Due to the errors below, we could not validate your formula. Please revise and try again.', 'wpforms-calculations' ),
					'ajaxFail'                         => esc_html__( 'Formula validation AJAX call error; Server response text:', 'wpforms-calculations' ),
					/* translators: %1$s - Variable name. */
					'validationVariableNotAllowed'     => esc_html__( 'Variable %1$s is not allowed. Only $Fx field variables are allowed.', 'wpforms-calculations' ),
					/* translators: %1$s - Field ID. */
					'validationFieldDoesntExist'       => esc_html__( 'Variable $F%1$s cannot be used. Field with ID %1$s does not exist.', 'wpforms-calculations' ),
					/* translators: %1$s - Total field ID, %2$s - Single Item field ID. */
					'validationFieldTotalInSingleItem' => esc_html__( 'Field Total (ID %1$s) could not be used in the Single Item field (ID %2$s).', 'wpforms-calculations' ),
					/* translators: %1$s - Field ID, %2$s - field type. */
					'validationFieldNotAllowed'        => esc_html__( 'Variable $F%1$s is not allowed. Field with ID %1$s has type "%2$s", which is not allowed.', 'wpforms-calculations' ),
					/* translators: %1$s - Field ID, %2$s - subfield. */
					'validationSubfieldNotAllowed'     => esc_html__( 'Variable $F%1$s_%2$s is not allowed. Field with ID %1$s does not have "%2$s" subfield.', 'wpforms-calculations' ),
					'field'                            => esc_html__( 'Field', 'wpforms-calculations' ),
					'fields'                           => esc_html__( 'Fields', 'wpforms-calculations' ),
					'thisFieldUsedInField'             => esc_html__( 'It\'s used in calculation in the following field:', 'wpforms-calculations' ),
					'thisFieldUsedInFields'            => esc_html__( 'It\'s used in calculations in the following fields:', 'wpforms-calculations' ),
				],
			]
		);
	}

	/**
	 * Calculation options.
	 *
	 * @since 1.0.0
	 *
	 * @param array         $field     Field data.
	 * @param WPForms_Field $field_obj Field object instance.
	 *
	 * @noinspection HtmlUnknownTarget
	 */
	public function advanced_options( $field, $field_obj ) {

		// Limit to our specific field types.
		if (
			! isset( $field['type'] ) ||
			! in_array( $field['type'], $this->helpers->get_fields_calculation_is_possible(), true )
		) {
			return;
		}

		$is_enabled = isset( $field['calculation_is_enabled'] ) ? $field['calculation_is_enabled'] : '0';

		$field_obj->field_element(
			'row',
			$field,
			[
				'slug'    => 'calculation_is_enabled',
				'class'   => '',
				'content' => $field_obj->field_element(
					'toggle',
					$field,
					[
						'slug'    => 'calculation_is_enabled',
						'value'   => $is_enabled,
						'desc'    => esc_html__( 'Enable Calculation', 'wpforms-calculations' ),
						'tooltip' => esc_html__( 'Enable calculations for this field.', 'wpforms-calculations' ),
					],
					false
				),
			]
		);

		$cheatsheet_link = sprintf(
			'<a href="%1$s" target="_blank" rel="noopener noreferrer" class="wpforms-calculations-cheatsheet-link">
				<i class="fa fa-file-text-o"></i><span>%2$s</span>
			</a>',
			esc_url( wpforms_utm_link( 'https://wpforms.com/calculations-formula-cheatsheet/', 'Calculations Formula', 'Calculations Cheatsheet' ) ),
			esc_html__( 'Cheatsheet', 'wpforms-calculations' )
		);

		$content = $field_obj->field_element(
			'label',
			$field,
			[
				'slug'          => 'calculation_code',
				'value'         => esc_html__( 'Formula', 'wpforms-calculations' ),
				'tooltip'       => esc_html__( 'Using your form fields as variables, enter your formula below and ensure there are no extra spaces. Validate your formula when finished.', 'wpforms-calculations' ),
				'after_tooltip' => $cheatsheet_link,
			],
			false
		)
		. $this->editor_options_markup( $field, $field_obj );

		$field_obj->field_element(
			'row',
			$field,
			[
				'slug'    => 'calculation_code',
				'class'   => $is_enabled ? '' : 'wpforms-hidden',
				'content' => $content,
			]
		);
	}

	/**
	 * Calculation editor options markup.
	 *
	 * @since 1.0.0
	 *
	 * @param array         $field     Field data.
	 * @param WPForms_Field $field_obj Field object instance.
	 */
	private function editor_options_markup( $field, $field_obj ) {

		return '<div class="wpforms-calculations-editor-collapsed">
			<div class="wpforms-calculations-editor-wrap">
				<div class="toolbar">
					<button type="button" class="button-insert-field">' . esc_html__( 'Insert Field', 'wpforms-calculations' ) . '</button>
					<button type="button" class="button-plus">+</button>
					<button type="button" class="button-plus">-</button>
					<button type="button" class="button-divide">/</button>
					<button type="button" class="button-multiply">*</button>
					<button type="button" class="button-br-open">(</button>
					<button type="button" class="button-br-close">)</button>
					<button type="button" class="button-point">.</button>
				</div>' .
				$field_obj->field_element(
					'textarea',
					$field,
					[
						'slug'  => 'calculation_code',
						'value' => isset( $field['calculation_code'] ) ? $field['calculation_code'] : '',
						'class' => [ 'wpforms-codemirror-editor' ],
					],
					false
				) .
				'<div class="wpforms-calculations-below-editor">
					<div class="wpforms-calculations-validate-formula-wrap">
						<button type="button" class="wpforms-calculations-validate-formula">' . esc_html__( 'Validate Formula', 'wpforms-calculations' ) . '</button>
						<span class="wpforms-calculations-validate-formula-status"></span>
					</div>
					<button type="button" class="wpforms-calculations-expand-editor">
						<i class="fa fa-expand"></i><span>' . esc_html__( 'Expand Editor', 'wpforms-calculations' ) . '</span>
					</button>
				</div>
				</div>
		</div>
		<div class="wpforms-calculations-editor-expanded"></div>';
	}
}
