/* global SignaturePad */

/**
 * @param signaturePad.fromDataURL
 */

// noinspection ES6ConvertVarToLetConst
/**
 * WPForms Signature function.
 *
 * @since 1.1.0
 */
// eslint-disable-next-line no-var
var WPFormsSignatures = window.WPFormsSignatures || ( function( document, window, $ ) {
	/**
	 * Private functions and properties.
	 *
	 * @since 1.1.0
	 *
	 * @type {Object}
	 */
	const __private = {

		/**
		 * Config contains all configuration properties.
		 *
		 * @since 1.1.0
		 *
		 * @type {Object}
		 */
		config: {

			// Window width to help process resize events.
			windowWidth: false,

			// Display ratio (e.g., retina or standard display).
			pixelRatio: Math.max( window.devicePixelRatio || 1, 1 ),

			// This is a placeholder for setInterval in some use cases.
			watching: false,

			// Time in ms to watch for disabled signatures.
			watchingRate: 300,
		},

		/**
		 * Returns the canvas element from jQuery signature object.
		 *
		 * Also make the necessary adjustments for high-res displays.
		 *
		 * @since 1.1.0
		 *
		 * @param {jQuery} $signature Signature object.
		 *
		 * @return {Object} Return the canvas element.
		 */
		getCanvas( $signature ) {
			const canvas = $signature.get( 0 );

			// This fixes issues with high-res/retina displays.
			canvas.width = canvas.offsetWidth * __private.config.pixelRatio;
			canvas.height = canvas.offsetHeight * __private.config.pixelRatio;

			canvas.getContext( '2d' ).scale( __private.config.pixelRatio, __private.config.pixelRatio );

			return canvas;
		},

		/**
		 * Sets all pixels in a canvas to black.
		 *
		 * @since 1.11.0
		 *
		 * @param {Object} canvas Signature canvas.
		 *
		 * @return {Object} Return the canvas with all pixels set to black.
		 */
		allPixelsToBlack( canvas ) {
			const ctx = canvas.getContext( '2d' ),
				imageData = ctx.getImageData( 0, 0, canvas.width, canvas.height ),
				data = imageData.data;

			for ( let i = 0; i < data.length; i += 4 ) {
				data[ i ] = 0; // Red.
				data[ i + 1 ] = 0; // Green.
				data[ i + 2 ] = 0; // Blue.
			}

			ctx.putImageData( imageData, 0, 0 );

			return canvas;
		},

		/**
		 * Crops a canvas so that all white space is removed.
		 *
		 * @since 1.1.0
		 *
		 * @param {Object} canvas Signature canvas.
		 *
		 * @return {string} Return data URL of the new cropped canvas.
		 */
		cropCanvas( canvas ) {
			// First duplicate the canvas to not alter the original.
			const croppedCanvas = document.createElement( 'canvas' ),
				croppedCtx = croppedCanvas.getContext( '2d' );

			croppedCanvas.width = canvas.width;
			croppedCanvas.height = canvas.height;

			croppedCtx.drawImage( canvas, 0, 0 );

			// Perform the cropping.
			let w = croppedCanvas.width,
				h = croppedCanvas.height,
				x,
				y,
				index,
				hasPixels = false;
			const pix = { x: [], y: [] },
				imageData = croppedCtx.getImageData( 0, 0, croppedCanvas.width, croppedCanvas.height );

			for ( y = 0; y < h; y++ ) {
				for ( x = 0; x < w; x++ ) {
					index = ( ( y * w ) + x ) * 4;

					if ( imageData.data[ index + 3 ] > 0 ) {
						pix.x.push( x );
						pix.y.push( y );
						hasPixels = true;
					}
				}
			}

			if ( ! hasPixels ) { // The canvas is empty.
				return '';
			}

			pix.x.sort( function( a, b ) {
				return a - b;
			} );

			pix.y.sort( function( a, b ) {
				return a - b;
			} );

			const n = pix.x.length - 1;

			w = pix.x[ n ] - pix.x[ 0 ];
			h = pix.y[ n ] - pix.y[ 0 ];

			const cut = croppedCtx.getImageData( pix.x[ 0 ], pix.y[ 0 ], w, h );

			croppedCanvas.width = w;
			croppedCanvas.height = h;

			croppedCtx.putImageData( cut, 0, 0 );

			// Return data URL of the new cropped canvas.
			return croppedCanvas.toDataURL();
		},

		/**
		 * Watches signatures that are currently hidden, detects when they
		 * become visible.
		 *
		 * @since 1.1.0
		 */
		watchSignatures() {
			let $allLoaded = true;

			$( '.wpforms-signature-canvas' ).each( function() {
				const $signature = $( this );

				if ( ! $signature.hasClass( 'loaded' ) ) {
					$allLoaded = false;

					// The signature is now visible.
					if ( ! $signature.is( ':hidden' ) ) {
						// Since we can see it, let's load it!
						app.loadSignature( $signature ); // eslint-disable-line no-use-before-define
					}
				}
			} );

			// If hidden signatures have been displayed, cease watching.
			if ( $allLoaded ) {
				clearInterval( __private.config.watching );
				__private.config.watching = false;
			}
		},
	};

	/**
	 * Elements holder.
	 *
	 * @since 1.11.0
	 *
	 * @type {Object}
	 */
	const el = {};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.1.0
	 *
	 * @type {Object}
	 */
	const app = {

		/**
		 * Start the engine.
		 *
		 * @since 1.1.0
		 */
		init() {
			el.$window = $( window );
			el.$document = $( document );
			el.$form = $( '.wpforms-form' );

			el.$document.on( 'wpformsReady', app.ready );
		},

		/**
		 * Initialize once the DOM is fully loaded.
		 *
		 * @since 1.1.0
		 */
		ready() {
			// Set the initial window width to filter for accurate resize events.
			__private.config.windowWidth = el.$window.width();

			// Initialize signature instances.
			app.loadSignatures( el.$document );

			// Define events listeners.
			app.events();

			// Enable the visibility watching to check if we have hidden signatures.
			__private.config.watching = setInterval( __private.watchSignatures, __private.config.watchingRate );
		},

		/**
		 * Events.
		 *
		 * @since 1.7.0
		 */
		events() {
			// Bind to form submission to convert the signature to be all black.
			el.$form.on( 'wpformsBeforeFormSubmit', app.wpformsBeforeFormSubmit );

			// Bind clear button to reset signature.
			el.$document.on( 'click', '.wpforms-signature-clear', function( event ) {
				event.preventDefault();

				const $btn = $( this );

				$btn.blur();
				app.resetSignature( $btn.parent().find( '.wpforms-signature-canvas' ), true );
			} );

			// Add handle for space bar to clear signature.
			el.$document.on( 'keydown', '.wpforms-signature-clear', function( event ) {
				if ( event.keyCode !== 32 ) {
					return;
				}

				event.preventDefault();

				const $btn = $( this );

				$btn.blur();
				app.resetSignature( $btn.parent().find( '.wpforms-signature-canvas' ), true );
			} );

			// Focus signature input on change.
			el.$document.on( 'wpforms:signature:change', '.wpforms-field-signature', function() {
				$( this )
					.find( '.wpforms-signature-input' )
					.trigger( 'focus' );
			} );

			// Initialize cloned Signatures inside the Repeater field.
			el.$document.on( 'wpformsRepeaterFieldCloneDisplay', function( e, $clone ) {
				app.loadSignatures( $clone );
			} );

			// Bind window resize to reset signatures.
			el.$window.resize( app.resetSignatures );
		},

		/**
		 * Before form submit, convert the signature to be all black.
		 *
		 * @since 1.11.0
		 *
		 * @param {Object} event Event object.
		 */
		wpformsBeforeFormSubmit( event ) {
			const $form = $( event.target );

			// Do not change the signature if it is inside a block.
			if ( ! $form.parents( '.wpforms-container.wpforms-block' ).length ) {
				return;
			}

			$form.find( '.wpforms-signature-canvas.loaded' ).each( function() {
				const $signature = $( this ),
					$wrap = $signature.closest( '.wpforms-field-signature' ),
					$input = $wrap.find( '.wpforms-signature-input' );

				// Convert color to black.
				$input.val( __private.cropCanvas( __private.allPixelsToBlack( $signature.get( 0 ) ) ) );
			} );
		},

		/**
		 * Finds, creates and loads each signature instance.
		 *
		 * @since 1.1.0
		 *
		 * @param {jQuery} $context Context container to search the Signatures.
		 */
		loadSignatures( $context ) {
			$context = $context?.length ? $context : el.$document;

			$context.find( '.wpforms-signature-canvas' ).each( function() {
				app.loadSignature( $( this ) );
			} );
		},

		/**
		 * Get the ink color for the signature.
		 *
		 * @since 1.10.0
		 *
		 * @param {jQuery} $container jQuery container object.
		 * @param {jQuery} $signature jQuery signature object.
		 *
		 * @return {string} Return the ink color.
		 */
		getInkColor( $container, $signature ) {
			const isBlock = typeof Array.from( $container[ 0 ].classList ).find( ( c ) => c.startsWith( 'wpforms-block-' ) ) !== 'undefined';
			const isModern = $container[ 0 ].classList.contains( 'wpforms-render-modern' );

			return isBlock && isModern
				? window.getComputedStyle( $container[ 0 ] ).getPropertyValue( '--wpforms-field-text-color' )
				: $signature.data( 'color' );
		},

		/**
		 * Creates and loads a single signature instance.
		 *
		 * @since 1.1.0
		 *
		 * @param {jQuery} $signature jQuery signature object.
		 */
		loadSignature( $signature ) {
			const $wrap = $signature.closest( '.wpforms-field-signature' );

			// Check if $wrap has parent <fieldset disabled> and if yes, return.
			if ( $wrap.parents( 'fieldset' ).is( ':disabled' ) ) {
				return;
			}

			if ( $signature.is( ':hidden' ) ) {
				return;
			}

			const $input = $wrap.find( '.wpforms-signature-input' ),
				$container = $wrap.closest( '.wpforms-container' ),
				canvas = __private.getCanvas( $signature ),
				color = app.getInkColor( $container, $signature );

			// Creates the signature instance.
			const signaturePad = new SignaturePad( canvas, {
				penColor: color,
			} );

			$signature
				.addClass( 'loaded' )
				.data( 'signaturePad', signaturePad );

			/**
			 * Bind to SignaturePad's afterUpdateStroke event
			 * and write signature data to a hidden input field.
			 *
			 * @see https://github.com/szimek/signature_pad#events
			 */
			signaturePad.addEventListener( 'afterUpdateStroke', function() {
				$input
					.val( __private.cropCanvas( canvas ) )
					.trigger( 'input change' )
					.valid();

				$wrap[ 0 ].dispatchEvent( new CustomEvent( 'wpforms:signature:change', {
					bubbles: true,
				} ) );
			} );
		},

		/**
		 * Reset signatures. This runs when the viewport size is changed.
		 *
		 * @since 1.1.0
		 */
		resetSignatures() {
			// If the viewport width has not changed, we do not need to reset.
			if ( __private.config.windowWidth === $( window ).width() ) {
				return;
			}

			$( '.wpforms-signature-canvas.loaded' ).each( function() {
				app.resetSignature( $( this ), false );
			} );
		},

		/**
		 * Reset the canvas for a signature.
		 *
		 * @since 1.1.0
		 * @since 1.5.0 Added a `clear` parameter.
		 *
		 * @param {jQuery}  $signature jQuery signature object.
		 * @param {boolean} clear      True if you need to clear the canvas.
		 */
		resetSignature( $signature, clear ) {
			const $wrap = $signature.closest( '.wpforms-field-signature' ),
				$input = $wrap.find( '.wpforms-signature-input' ),
				signaturePad = $signature.data( 'signaturePad' ),
				value = signaturePad.toDataURL( 'image/svg+xml' );

			if ( clear ) {
				signaturePad.clear();
				$input.val( '' );
			} else {
				// Properly scale canvas.
				__private.getCanvas( $signature );
				signaturePad.fromDataURL( value	);
			}

			$input.trigger( 'input change' );

			// Check if the signature is hidden.
			if ( $signature.is( ':hidden' ) ) {
				$signature.removeClass( 'loaded' );

				// Enable visibility watching if not already running.
				if ( ! __private.config.watching ) {
					__private.config.watching = setInterval( __private.watchSignatures, __private.config.watchingRate );
				}
			}
		},
	};

	// Provide access to public functions/properties.
	return app;
}( document, window, jQuery ) );

// Initialize.
WPFormsSignatures.init();
